/*
 * The routines here encapsulate the state transition knowledge of the
 * FMS and what to do at each transition.
 * There are basically three states to the FMS:
 *  MAPPING - the fabric is or is about to be mapped.
 *  RESOLVING - topologically discovered xbars are being resolved against 
 *	monitoring card information
 *  VERIFYING - this is the idle state, verifies are active at this point
 *  QUIET - a user-imposed state to silence all fma traffic on the fabric
 */

#include "libfma.h"
#include "lf_fabric.h"

#include "fms.h"
#include "fms_fabric.h"
#include "fms_fma.h"
#include "fms_error.h"
#include "fms_state.h"
#include "fms_notify.h"
#include "fms_resolve.h"
#include "fms_settings.h"

/*
 * Initialize state variables
 */
void
fms_init_state_vars()
{
  struct fms_state *fsp;

  LF_CALLOC(fsp, struct fms_state, 1);
  F.state = fsp;

  fsp->state = FMS_STATE_IDLE;
  return;

 except:
  fms_perror_exit(1);
}

/*
 * Called just after a map has been sent.
 */
void
fms_state_map_sent()
{
  struct fms_state *fsp;
  struct fms_settings *fsetp;
  struct lf_fabric *fp;

  fms_notify(FMS_EVENT_DEBUG, "State: map sent");

  fsp = F.state;	/* get state pointer */
  fsetp = F.settings;
  fp = F.fabvars->fabric;

  /*
   * Make sure no map request pending since we just sent one
   * (just to validate state machine is not wonky)
   */
  if (fms_map_request_pending()) {
    LF_ERROR(("Map just sent and map request pending?"));
  }

  /*
   * See if there is any resolving that needs to be done
   */
  if (fms_fabric_needs_resolution()) {
    fms_notify(FMS_EVENT_INFO, "State: starting fabric resolution");
    fms_resolve_fabric();
    fsp->state = FMS_STATE_RESOLVING;

  /* If not, we are simply in verify mode */
  } else {
    fsp->state = FMS_STATE_VERIFYING;
  }
  return;

 except:
  fms_perror_exit(1);
}

/*
 * We have made a change to the fabric, possibly because of a new FMA
 * coming online with new information.
 */
void
fms_state_fabric_changed(
  enum fms_state_change what_changed)
{
  struct fms_state *fsp;
  struct lf_fabric *fp;

  fsp = F.state;	/* get state pointer */
  fp = F.fabvars->fabric;

  fms_notify(FMS_EVENT_DEBUG, "State: fabric has changed: what=%d.",
             what_changed);

  /*
   * Take some actions based on current state
   */
  switch (fsp->state) {

    /* If resolving, stop */
    case FMS_STATE_RESOLVING:
      fms_notify(FMS_EVENT_INFO, "State: stopping fabric resolution");
      fms_stop_fabric_resolution();
      break;

    default:
      break;
  }

  lf_build_xbar_array(fp);      /* this brings fp->num_xbars up-to-date */

  /* mark all FMAs as needing a new map */
  fms_fma_invalidate_map();

  /* schedule a new map to be pushed - the actual topo map will be generated
   * at the last possible moment.
   */
  fms_schedule_map_push();

  /* wherever we were, state is now SEND_MAP */
  fsp->state = FMS_STATE_SEND_MAP;
}

/*
 * We have been informed that our current concept of the fabric is incorrect.
 * Ask an FMA out there to generate a map for us.
 */
void
fms_state_map_is_invalid(
  struct fms_fma_desc *adp)
{
  struct fms_state *fsp;
  struct lf_fabric *fp;

  fms_notify(FMS_EVENT_DEBUG, "State: map reported invalid");

  fsp = F.state;	/* get state pointer */
  fp = F.fabvars->fabric;

  switch (fsp->state) {
    case FMS_STATE_VERIFYING:
      break;

    /* If resolving, stop */
    case FMS_STATE_RESOLVING:
      fms_notify(FMS_EVENT_INFO, "State: stopping fabric resolution");
      fms_stop_fabric_resolution();
      break;

    default:
      break;
  }

  fms_cancel_map_push();	/* cancel any push since it is invalid */

  /* save mapper hint if provided */
  if (adp != NULL) {
    FMS_FABRIC(fp)->invalid_map_reporter = adp;
  }

  fms_schedule_map_request();   /* request a new map */

  fsp->state = FMS_STATE_GET_MAP;	/* getting map */
}

/*
 * Called when all possible switch resolution has completed.
 */
void
fms_state_fabric_resolution_complete()
{
  struct fms_state *fsp;
  struct lf_fabric *fp;

  fsp = F.state;	/* get state pointer */
  fp = F.fabvars->fabric;

  fms_notify(FMS_EVENT_INFO, "State: fabric resolution complete");

  /* treat this as a fabric change */
  fsp->state = FMS_STATE_IDLE;
  fms_state_fabric_changed(FMS_STATE_FABRIC_CHANGE);
}
